package furny;

import furny.ga.FurnGA;
import furny.ga.FurnLayoutIndividual;
import furny.ga.PseudoSpace;
import furny.ga.operators.RealMutationOp;
import furny.ga.operators.SegmentCutCrossoverOp;
import furny.ga.rules.FurnitureIntersectionRule;
import furny.ga.rules.RoomContainsRule;
import ga.core.GA;
import ga.core.algorithm.automatic.SGAGeneration;
import ga.core.algorithm.util.PopulationUtil;
import ga.core.goperators.ICrossoverOp;
import ga.core.goperators.IMutationOp;
import ga.core.individual.IIndividualFactory;
import ga.core.individual.TemplateIndividualFactory;
import ga.core.individual.population.ArrayListPopulation;
import ga.core.individual.population.IPopulation;
import ga.core.individual.population.KMeansClusterPopulation;
import ga.core.selection.TournamentSelector;
import ga.core.validation.GAContext;
import ga.core.validation.RuleValidator;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.Locale;
import java.util.logging.Level;
import java.util.logging.Logger;

public class FurnyBench {

  // the logger for this class
  private static final Logger LOGGER = Logger.getLogger(FurnyBench.class
      .getName());

  public static void main(final String[] args) {

    final int popSize = 100;
    final int pMutate = 5;
    final int pAddFurniture = 5;
    final int pRemoveFurniture = 5;
    final int pCrossover = 95;
    final int tournamentSize = 10;

    final int clusterPopSize = 20;
    final int clusterCount = 10;

    final int generations = 100;

    final boolean useClustering = true;
    final boolean useEliteStrategy = false;

    final int wID = 2;
    final int wTranslate = 40;
    final int wRotate = 2;

    final boolean performanceLogging = false;

    final double[][] csvData = new double[generations + 1][];

    final String target = "{(380,114,-18,3),(388,-67,-94,0)}";
    // final String target =
    // "{(393,108,147,3),(447,-111,-189,0),(445,201,-162,1),(438,-131,89,2)}";

    if (performanceLogging) {
      LOGGER.getParent().setLevel(Level.SEVERE);
    }

    final GAContext context = new GAContext();
    context.put(GA.KEY_VALIDATION_SPACE, new PseudoSpace(5f, 5f));

    context.put(GA.KEY_GENOME_MIN_LENGTH, 2);
    context.put(GA.KEY_GENOME_MAX_LENGTH, 6);
    context.put(FurnGA.KEY_WEIGHT_ID, 4);
    context.put(FurnGA.KEY_WEIGHT_TRANSLATION, 2);
    context.put(FurnGA.KEY_WEIGHT_ROTATION, 1);

    final RuleValidator<FurnLayoutIndividual> validator = new RuleValidator<FurnLayoutIndividual>();

    validator.addRule(new FurnitureIntersectionRule());
    validator.addRule(new RoomContainsRule());

    final FurnLayoutIndividual template = new FurnLayoutIndividual(context);
    final IIndividualFactory<FurnLayoutIndividual> factory = new TemplateIndividualFactory<FurnLayoutIndividual>(
        template);

    final IPopulation<FurnLayoutIndividual> population;
    final FurnLayoutEvaluator evaluator = new FurnLayoutEvaluator(target,
        context);
    final TournamentSelector<FurnLayoutIndividual> selector;
    final ICrossoverOp<FurnLayoutIndividual> crossOverOp;
    final IMutationOp<FurnLayoutIndividual> mutationOp;

    template.initRandomly();

    // //////////////////
    if (useClustering) {
      population = new KMeansClusterPopulation<FurnLayoutIndividual>(factory,
          clusterPopSize, clusterCount);
    } else {
      population = new ArrayListPopulation<FurnLayoutIndividual>(factory,
          popSize);
    }
    selector = new TournamentSelector<FurnLayoutIndividual>(tournamentSize);
    // crossOverOp = new OnePointCrossoverOp(pCrossover);
    crossOverOp = new SegmentCutCrossoverOp(pCrossover);
    // mutationOp = new FurnLayoutMutationOp(pMutate);
    mutationOp = new RealMutationOp(pMutate, wID, wTranslate, wRotate,
        pAddFurniture, pRemoveFurniture);
    // /////////////////

    final SGAGeneration<FurnLayoutIndividual> algorithm = new SGAGeneration<FurnLayoutIndividual>(
        population, evaluator, selector, mutationOp, crossOverOp,
        useEliteStrategy, context);

    algorithm.setValidator(validator);

    algorithm.init();

    int successGeneration = -1;

    for (int i = 0; i <= generations; i++) {
      if (LOGGER.isLoggable(Level.INFO)) {
        LOGGER.info("Generation " + i);
      }
      // LOGGER.info("Population " + population.toString());

      final long time = System.nanoTime();
      algorithm.step();
      final long t2 = (System.nanoTime() - time) / 1000000;

      if (population.size() < popSize) {
        LOGGER.warning("The population is getting smaller!!");
      }

      if (LOGGER.isLoggable(Level.INFO)) {
        final FurnLayoutIndividual fittest = population.getFittestIndividual();
        LOGGER.info("Fittest: " + fittest.getDebugString());

        if ((fittest.getFitness() == 100 || population.getIndividuals()
            .contains(evaluator.getTargetIndividual()))
            && successGeneration < 0) {
          successGeneration = i;
        }

        final double[] mmm = PopulationUtil.getMinMeanMaxFitness(population);
        LOGGER.info(String.format(Locale.ENGLISH,
            "Min: %.2f Mean: %.2f Max: %.2f", mmm[0], mmm[1], mmm[2]));

        csvData[i] = mmm;
      } else {
        System.out.println(t2 + "ms ");
      }
    }

    if (performanceLogging) {
      final double[] mmm = PopulationUtil.getMinMeanMaxFitness(population);
      System.out.println(String.format(Locale.ENGLISH,
          "Min: %.2f Mean: %.2f Max: %.2f", mmm[0], mmm[1], mmm[2]));

      final FurnLayoutIndividual fittest = population.getFittestIndividual();
      System.out.println("Fittest: " + fittest);
    }
    if (successGeneration > 0) {
      LOGGER.info("Reached perfect individual after " + successGeneration
          + " generations");
    }
    LOGGER.info("done.");

    writeCSV(csvData);
  }

  private static void writeCSV(final double[][] data) {
    final File csvFile = new File("fitness.csv");
    BufferedWriter bw = null;
    try {
      bw = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(
          csvFile)));

      bw.write("i;fit_min;fit_mean;fit_max\n");

      for (int i = 0; i < data.length; i++) {
        try {
          bw.write(String.format("%d;%,.2f;%,.2f;%,.2f;\n", i, data[i][0],
              data[i][1], data[i][2]));
        } catch (final Exception e) {
          e.printStackTrace();
        }

      }

    } catch (final IOException e) {
      e.printStackTrace();
    } finally {
      try {
        bw.close();
      } catch (final IOException e) {
      }
    }
  }
}
